using System;
using System.Collections.Generic;
using System.Linq;

using LibBSP;

namespace Decompiler {
	/// <summary>
	/// Helper class containing methods for working with <see cref="StaticModel"/> objects.
	/// </summary>
	public static class StaticModelExtensions {

		/// <summary>
		/// Parse the properties of this <see cref="StaticModel"/> into an <see cref="Entity"/> object.
		/// </summary>
		/// <param name="model">This <see cref="StaticModel"/>.</param>
		/// <returns><see cref="Entity"/> representation of this <see cref="StaticModel"/>.</returns>
		public static Entity ToEntity(this StaticModel model) {
			string name = model.name.Replace("//", "/").Trim('/').ToLower();
			string classname;
			if (modelToEntityMap.ContainsKey(name)) {
				classname = modelToEntityMap[name];
			} else {
				classname = "static_model";
			}
			Entity entity = new Entity(classname);
			entity["model"] = model.name;
			entity.origin = model.origin;
			entity.angles = model.angles;
			entity["scale"] = model.scale.ToString();
			entity["angle"] = "0";
			return entity;
		}

		/// <summary> Mapping of all .TIK files to entity classes in MoHAA, including Spearhead and Breakthrough. </summary>
		/// <remarks>
		/// I really wish there were a better way to do this mapping, but there doesn't seem to be. Unless I want to include
		/// a Tiki parser with the decompiler and find and read all the TIK files referenced within a map and read the entity
		/// name from there, this is the best way to do this. Maybe this list should be in a resource file instead.
		/// </remarks>
		private static readonly Dictionary<string, string> modelToEntityMap = new Dictionary<string, string>() {
			{ "animate/body_us01.tik", "static_deadbodies_dday-us-base-pose-1" },
			{ "animate/body_us02.tik", "static_deadbodies_dday-us-base-pose-2" },
			{ "animate/body_us03.tik", "static_deadbodies_dday-us-base-pose-3" },
			{ "equipment/german_prop_helmet.tik", "static_equipment_german-prop-helmet" },
			{ "furniture/armchairposh.tik", "static_furniture_armchairposh" },
			{ "furniture/bunkerchair.tik", "static_furniture_bunkerchair" },
			{ "furniture/bunkerchairstool.tik", "static_furniture_bunkerchairstool" },
			{ "furniture/bunkerchairstool_d.tik", "static_furniture_bunkerchairstool" },
			{ "furniture/bunkerchair_d.tik", "static_furniture_bunkerchair-d" },
			{ "furniture/cardchair.tik", "static_furniture_card-chair" },
			{ "furniture/grillchair.tik", "static_furniture_grillchair" },
			{ "furniture/interrogationchair.tik", "static_furniture_interrogation-chair" },
			{ "furniture/luxurychair.tik", "static_furniture_luxurychair" },
			{ "furniture/luxurychair_d.tik", "static_furniture_luxurychair-d" },
			{ "furniture/luxuryfootrest.tik", "static_furniture_luxuryfootrest" },
			{ "furniture/luxuryfootrest_d.tik", "static_furniture_luxuryfootrest-d" },
			{ "furniture/roundedchair.tik", "static_furniture_roundedchair" },
			{ "furniture/roundedchair_d.tik", "static_furniture_roundedchair-d" },
			{ "furniture/simplechair.tik", "static_furniture_simplechair" },
			{ "furniture/simplestool.tik", "static_furniture_simplestool" },
			{ "furniture/stoolposh.tik", "static_furniture_stoolposh" },
			{ "furniture/table.tik", "static_furniture_table" },
			{ "furniture/woodchair.tik", "static_furniture_woodchair" },
			{ "furniture/woodchair_d.tik", "static_furniture_woodchair-d" },
			{ "items/bangalore.tik", "static_item_bangalore" },
			{ "items/battery.tik", "static_item_battery" },
			{ "items/bratwurst.tik", "static_item_bratwurst" },
			{ "items/cigarette.tik", "static_items_cigarette" },
			{ "items/documents1a.tik", "static_items_documents1a(0.0" },
			{ "items/documents1b.tik", "static_items_documents1b(0.0" },
			{ "items/documents1c.tik", "static_items_documents1c(0.0" },
			{ "items/documents1d.tik", "static_items_documents1d(0.0" },
			{ "items/documents2a.tik", "static_items_documents2a(0.0" },
			{ "items/documents2b.tik", "static_items_documents2b(0.0" },
			{ "items/documents2c.tik", "static_items_documents2c(0.0" },
			{ "items/documents2d.tik", "static_items_documents2d(0.0" },
			{ "items/explosive.tik", "static_item_explosive" },
			{ "items/explosive2.tik", "static_item_explosive2" },
			{ "items/flaregun.tik", "static_item_flaregun" },
			{ "items/gasmask.tik", "static_item_gasmask" },
			{ "items/officer_uniform.tik", "static_items_officer-uniform(0.0" },
			{ "items/papers_open.tik", "static_item_papers-open" },
			{ "items/papers_open2.tik", "static_item_papers2-open" },
			{ "items/torpedo.tik", "static_uboat_torpedo" },
			{ "items/wirecutters.tik", "static_item_wirecutters" },
			{ "lights/ceilinglamp1.tik", "static_lamp_ceilinglamp1" },
			{ "lights/chandalier.tik", "static_lamp_chandalier" },
			{ "lights/fattablelamp.tik", "static_lamp_fattablelamp" },
			{ "lights/glasslamp.tik", "static_lamp_glasslamp" },
			{ "lights/hang3lamp.tik", "static_lamp_hang3lamp" },
			{ "lights/hang5lamp.tik", "static_lamp_hang5lamp" },
			{ "lights/hanglamp.tik", "static_lamp_hanglamp" },
			{ "lights/lightbulb_caged.tik", "static_lamp_lightbulb-caged" },
			{ "lights/lightbulb_caged_red-lit.tik", "static_lamp_lightbulb-caged-red-lit" },
			{ "lights/lightbulb_caged_red.tik", "static_lamp_lightbulb-caged-red" },
			{ "lights/tablelamp.tik", "static_lamp_tablelamp" },
			{ "lights/tiffanylamp.tik", "static_lamp_tiffanylamp" },
			{ "lights/wallsconce-candle.tik", "static_lamp_wallsconce-candle" },
			{ "lights/wallsconce-cutglass.tik", "static_lamp_wallsconce-cutglass" },
			{ "lights/wallsconce-frosted.tik", "static_lamp_wallsconce-frosted" },
			{ "lights/wallsconce-single.tik", "static_lamp_wallsconce-single" },
			{ "miscobj/alarmswitch.tik", "static_equipment_alarmswitch" },
			{ "miscobj/barbwire_long_pulse.tik", "static_obstacle_barbwire-long-pulsing" },
			{ "miscobj/beef.tik", "static_food_frozensidebeef" },
			{ "miscobj/canteen.tik", "static_attached-canteen(0" },
			{ "miscobj/cardhand01.tik", "static_items_cardhand01" },
			{ "miscobj/cardhand02.tik", "static_items_cardhand02" },
			{ "miscobj/cardpile.tik", "static_items_cardpile" },
			{ "miscobj/cardsingle.tik", "static_items_singlecard" },
			{ "miscobj/clipboard.tik", "static_items_clipboard(0.0" },
			{ "miscobj/crane.tik", "static_equipment_crane" },
			{ "miscobj/detonator.tik", "static_equipment_detonator" },
			{ "miscobj/detpack.tik", "static_detpack" },
			{ "miscobj/electrical_switch_nopulse.tik", "static_equipment_electric-switch-nopulse" },
			{ "miscobj/electrical_switch_pulse.tik", "static_equipment_electrical-switch-pulse" },
			{ "miscobj/hammer.tik", "static_hammer" },
			{ "miscobj/helmet_hand.tik", "static_equipment_crate-carry" },
			{ "miscobj/hidden_cabinet_a.tik", "static_furniture_hidden-wardrobe" },
			{ "miscobj/hidden_cabinet_b.tik", "static_furniture_hidden-cabinet-dark" },
			{ "miscobj/hidden_cabinet_c.tik", "static_furniture_hidden-cabinet-tall" },
			{ "miscobj/move_hedgehog.tik", "static_obstacle_move-hedgehog" },
			{ "miscobj/pencil.tik", "static_items_pencil" },
			{ "miscobj/phone_roundbase.tik", "static_decor_phone-roundbase" },
			{ "miscobj/phone_squarebase.tik", "static_decor_phone-squarebase" },
			{ "miscobj/picklock.tik", "static_equipment_picklock" },
			{ "miscobj/radar.tik", "static_equipment_radarstation" },
			{ "miscobj/radio_civilian.tik", "static_decor_radio-civilian" },
			{ "miscobj/radio_military.tik", "static_equipment_radio-military" },
			{ "miscobj/sledgehammer.tik", "static_equipment_sledgehammer" },
			{ "miscobj/subcrane.tik", "static_equipment_subcrane" },
			{ "miscobj/trainswitch.tik", "static_equipment_trainswitch" },
			{ "miscobj/trainswitch_pulsating.tik", "static_equipment_trainswitch-pulsating" },
			{ "miscobj/welding_mask.tik", "static_equipment_welding-mask" },
			{ "miscobj/welding_tank.tik", "static_equipment_welding-tank" },
			{ "miscobj/welding_torch.tik", "static_equipment_welding-torch" },
			{ "miscobj/wrench.tik", "static_wrench" },
			{ "static/30cal_crate.tik", "static_item_30cal_crate" },
			{ "static/45cal_crate.tik", "static_item_45cal_crate" },
			{ "static/50cal_crate.tik", "static_item_50cal_crate" },
			{ "static/88shell.tik", "static_88shell" },
			{ "static/alarmbell.tik", "static_miscobj_alarmbell" },
			{ "static/alarmswitch.tik", "static_items_alarmswitch" },
			{ "static/ballpeen_hammer.tik", "static_item_ballpeen-hammer" },
			{ "static/banquet_table.tik", "static_furniture_banquet-table" },
			{ "static/barbwire.tik", "static_obstacle_barbwire" },
			{ "static/barbwire_long.tik", "static_obstacle_barbwire-long" },
			{ "static/barbwire_long_one_post.tik", "static_obstacle_barbwire-long-one-post" },
			{ "static/barbwire_long_two_post.tik", "static_obstacle_barbwire-long-two-post" },
			{ "static/barbwire_one_post.tik", "static_obstacle_barbwire-one-post" },
			{ "static/barbwire_post.tik", "static_obstacle_barbwire-post" },
			{ "static/barbwire_tetra_post.tik", "static_obstacle_barbwire-tetra-post" },
			{ "static/barbwire_two_post.tik", "static_obstacle_barbwire-two-post" },
			{ "static/basket1.tik", "static_basket1" },
			{ "static/bathroomsink.tik", "static_furniture_bathroomsink" },
			{ "static/bathtub.tik", "static_furniture_bathtub" },
			{ "static/bathtubwithshowerhead.tik", "static_furniture_bathtubwithshowerhead" },
			{ "static/bigbed.tik", "static_furniture_bigbed" },
			{ "static/bigcircletable.tik", "static_furniture_bigcircletable" },
			{ "static/bigfilecabinet.tik", "static_furniture_bigfilecabinet" },
			{ "static/bighutch.tik", "static_furniture_bighutch" },
			{ "static/bodies_tarp.tik", "static_deadbodies_bodies-under-tarp" },
			{ "static/body_german01.tik", "static_deadbodies_dday-german-base-pose-1" },
			{ "static/body_us01.tik", "static_deadbodies_dday-us-base-pose-1" },
			{ "static/body_us02.tik", "static_deadbodies_dday-us-base-pose-2" },
			{ "static/body_us03.tik", "static_deadbodies_dday-us-base-pose-3" },
			{ "static/body_us04.tik", "static_deadbodies_dday-us-base-pose-4" },
			{ "static/body_us05.tik", "static_deadbodies_dday-us-base-pose-5" },
			{ "static/body_us06.tik", "static_deadbodies_dday-us-base-pose-6" },
			{ "static/body_usvest02.tik", "static_deadbodies_dday-pose-1" },
			{ "static/body_usvest03.tik", "static_deadbodies_dday-pose-3" },
			{ "static/body_usvest04.tik", "static_deadbodies_dday-pose-4" },
			{ "static/body_usvest05.tik", "static_deadbodies_dday-pose-5" },
			{ "static/body_usvest06.tik", "static_deadbodies_dday-pose-6" },
			{ "static/bookcase.tik", "static_furniture_bookcase" },
			{ "static/bottle_large.tik", "static_decor_bottle-large" },
			{ "static/bottle_small.tik", "static_decor_bottle-small" },
			{ "static/bshelf-tall-thin-damaged.tik", "static_furniture_bshelf-tall-thin-damaged" },
			{ "static/bshelf-tall-thin.tik", "static_furniture_bshelf-tall-thin" },
			{ "static/bucket.tik", "static_decor_bucket" },
			{ "static/bunkbed.tik", "static_furniture_bunkbed" },
			{ "static/bunkerbench.tik", "static_furniture_bunkerbench" },
			{ "static/bunkerfan.tik", "static_equipment_bunkerfan" },
			{ "static/bunkershelves.tik", "static_furniture_bunkershelves" },
			{ "static/bunkerstove.tik", "static_equipment_bunkerstove" },
			{ "static/bunkerstove_ns.tik", "static_equipment_bunkerstove" },
			{ "static/bunkertable.tik", "static_furniture_bunkertable" },
			{ "static/bunkertable_ns.tik", "static_furniture_bunkertable" },
			{ "static/bush_buckthorn.tik", "static_natural_bush_buckthorn" },
			{ "static/bush_full.tik", "static_natural_bush_full" },
			{ "static/bush_full_heavyweather.tik", "static_bush_full_heavyweather" },
			{ "static/bush_ivycover.tik", "static_natural_bush_ivycover" },
			{ "static/bush_lowlyingleaves.tik", "static_natural_bush_lowlyingleaves" },
			{ "static/bush_regularbush.tik", "static_natural_bush_regularbush" },
			{ "static/bush_sappling.tik", "static_natural_bush_sappling" },
			{ "static/bush_sparseshrub.tik", "static_natural_bush_sparseshrub" },
			{ "static/bush_winter_denseshrub.tik", "static_natural_bush_winter-denseshrub" },
			{ "static/bush_winter_fullshrub.tik", "static_natural_bush_winter-fullshrub" },
			{ "static/cabinet_dark.tik", "static_furniture_cabinet-dark" },
			{ "static/cabinet_large.tik", "static_furniture_cabinet-large" },
			{ "static/cabinet_small.tik", "static_furniture_cabinet-small" },
			{ "static/cabinet_tall.tik", "static_furniture_cabinet-tall" },
			{ "static/card_table.tik", "static_furniture_card-table" },
			{ "static/cart.tik", "static_furniture_cart" },
			{ "static/cello.tik", "static_furniture_cello" },
			{ "static/churchpew.tik", "static_furniture_churchpew" },
			{ "static/circulartable.tik", "static_furniture_circulartable" },
			{ "static/coatrack.tik", "static_furniture_coatrack" },
			{ "static/coat_hanging_stand.tik", "static_item_coat-hanging-stand" },
			{ "static/conningtower.tik", "static_submodels_conningtower" },
			{ "static/corona_orange.tik", "static_corona_orange" },
			{ "static/corona_reg.tik", "static_corona_reg" },
			{ "static/corona_util.tik", "static_corona_util" },
			{ "static/cot.tik", "static_furniture_cot" },
			{ "static/cot_nonstatic.tik", "static_furniture_cot" },
			{ "static/cratelid1.tik", "static_item_cratelid1" },
			{ "static/cratelid2.tik", "static_item_cratelid2" },
			{ "static/curtain.tik", "static_furniture_curtain" },
			{ "static/curtain_dirty.tik", "static_furniture_curtain-dirty" },
			{ "static/deckofcards.tik", "static_decor_cards" },
			{ "static/desk.tik", "static_furniture_desk" },
			{ "static/dish.tik", "static_item_dish" },
			{ "static/doublemetalbed.tik", "static_furniture_doublemetalbed" },
			{ "static/drawing_compass.tik", "static_equipment_drawing-compass" },
			{ "static/dresser.tik", "static_furniture_dresser" },
			{ "static/d_backhull.tik", "static_vehicles_uboat-destroyedback" },
			{ "static/d_fronthull.tik", "static_vehicles_uboat-destroyedfront" },
			{ "static/exp_crate1.tik", "static_item_exp_crate1" },
			{ "static/exp_crate1a.tik", "static_item_exp_crate1a" },
			{ "static/exp_crate2.tik", "static_item_exp_crate2" },
			{ "static/exp_crate2a.tik", "static_item_exp_crate2a" },
			{ "static/exp_crate3.tik", "static_item_exp_crate3" },
			{ "static/exp_crate3a.tik", "static_item_exp_crate3a" },
			{ "static/filter_thing.tik", "static_furniture_filter-thing" },
			{ "static/fire.tik", "static_fx_fire(0.5" },
			{ "static/fireplaceset.tik", "static_furniture_fireplaceset" },
			{ "static/flourbag-bent.tik", "static_item_flourbag-bent" },
			{ "static/flourbag-single.tik", "static_item_flourbag-single" },
			{ "static/flourbag-stack.tik", "static_item_flourbag-stack" },
			{ "static/flowerpainting.tik", "static_furniture_flowerpainting" },
			{ "static/flowerplate.tik", "static_item_flowerplate" },
			{ "static/fragcrate1.tik", "static_item_fragcrate1" },
			{ "static/fragcrate1a.tik", "static_item_fragcrate1a" },
			{ "static/fryingpan.tik", "static_item_fryingpan" },
			{ "static/german_lantern.tik", "static_lamp_geman-lantern" },
			{ "static/german_rations_1.tik", "static_equipment_german-rations-1" },
			{ "static/german_rations_2.tik", "static_equipment_german-rations-2" },
			{ "static/grandfathers_clock.tik", "static_furniture_grandfathers-clock" },
			{ "static/grandfathers_clock_d.tik", "static_furniture_grandfathers-clock-d" },
			{ "static/halfcouch.tik", "static_furniture_couch-half" },
			{ "static/haystack.tik", "static_farm_haystack" },
			{ "static/heat_crate.tik", "static_item_heat_crate" },
			{ "static/hedgehog.tik", "static_obstacle_hedgehog" },
			{ "static/hedgehog_cluster_three.tik", "static_obstacle_hedgehog-cluster-three" },
			{ "static/hedgehog_cluster_two.tik", "static_obstacle_hedgehog-cluster-two" },
			{ "static/hedgehog_cluster_two_seperated.tik", "static_obstacle_hedgehog-cluster-two-seperated" },
			{ "static/higginsxtrahull.tik", "static_vehicle_allied_higgins-boat-thickhull" },
			{ "static/hopechest.tik", "static_furniture_hopechest" },
			{ "static/indycrate.tik", "static_item_indycrate" },
			{ "static/jug.tik", "static_decor_jug" },
			{ "static/large_desk.tik", "static_furniture_largedesk" },
			{ "static/librarytable.tik", "static_furniture_librarytable(0.5" },
			{ "static/librarytabledestroyed.tik", "static_furniture_librarytabledestroyed(0.5" },
			{ "static/lightbulb_caged.tik", "static_lamp_lightbulb-caged" },
			{ "static/lightbulb_covered_nowire.tik", "static_lamp_lightbulb-covered-nowire" },
			{ "static/lightbulb_covered_on_wire.tik", "static_lamp_lightbulb-covered-on-wire" },
			{ "static/lightbulb_covered_on_wire_short.tik", "static_lamp_lightbulb-covered-on-wire-short" },
			{ "static/lightbulb_desertcage.tik", "static_lamp_lightbulb-desertcage" },
			{ "static/lightbulb_nowire.tik", "static_lamp_lightbulb-nowire" },
			{ "static/lightbulb_on_wire.tik", "static_lamp_lightbulb-on-wire" },
			{ "static/lightbulb_on_wire_short.tik", "static_lamp_lightbulb-on-wire-short" },
			{ "static/lightbulb_on_wire_s_ns.tik", "static_lamp_lightbulb-on-wire-short" },
			{ "static/lightpost_globe.tik", "static_lamp_lightpost-globe" },
			{ "static/lightpost_globe_winter.tik", "static_lamp_lightpost-globe-winter" },
			{ "static/lightpost_sidemounted.tik", "static_lamp_lightpost-sidemounted" },
			{ "static/lightpost_sidemounted_winter.tik", "static_lamp_lightpost-sidemounted-winter" },
			{ "static/lightpost_triple.tik", "static_lamp_lightpost-triple" },
			{ "static/lightpost_tripler_winter.tik", "static_lamp_lightpost-triple-winter" },
			{ "static/locker.tik", "static_furniture_locker" },
			{ "static/locker_ns.tik", "static_furniture_locker(0.5" },
			{ "static/loveseat.tik", "static_furniture_loveseat" },
			{ "static/loveseatdestroyed.tik", "static_furniture_loveseatdestroyed" },
			{ "static/lowerhull.tik", "static_vehicles_uboat-lowerhull" },
			{ "static/messkit.tik", "static_item_messkit" },
			{ "static/metalbench.tik", "static_furniture_metalbench" },
			{ "static/metaldesk.tik", "static_furniture_metaldesk" },
			{ "static/metaldesk_ns.tik", "static_furniture_metaldesk" },
			{ "static/metal_file.tik", "static_item_metal-file" },
			{ "static/metal_funnel.tik", "static_item_metal-funnel" },
			{ "static/metal_mug.tik", "static_item_metal-mug" },
			{ "static/mg42ammoboxwbelt.tik", "static_item_mg42ammobox-wbelt" },
			{ "static/microphone.tik", "static_microphone" },
			{ "static/microphone_ns.tik", "static_microphone-ns" },
			{ "static/minepole.tik", "static_obstacle_minepole" },
			{ "static/nazi_crate.tik", "static_item_nazicrate" },
			{ "static/night_stand.tik", "static_furniture_night-stand" },
			{ "static/oldladypainting.tik", "static_furniture_oldladypainting" },
			{ "static/opeltruck_hoodopen.tik", "static_vehicle_german_opeltruck-hoodopen" },
			{ "static/ornaterectable.tik", "static_furniture_ornaterectable" },
			{ "static/ornaterectable_d.tik", "static_furniture_ornaterectable-d" },
			{ "static/oval_table.tik", "static_furniture_oval-table" },
			{ "static/peetrough.tik", "static_furniture_peetrough" },
			{ "static/pens.tik", "static_item_pens" },
			{ "static/pew_damaged.tik", "static_furniture_pew-damaged" },
			{ "static/phonograph.tik", "static_furniture_phonograph" },
			{ "static/phonograph_ns.tik", "static_furniture_phonograph" },
			{ "static/piano.tik", "static_furniture_piano" },
			{ "static/piano_d.tik", "static_furniture_piano-damaged" },
			{ "static/piperplane.tik", "static_vehicle_piperplane(0.5" },
			{ "static/pitcher.tik", "static_decor_pitcher" },
			{ "static/pitchfork.tik", "static_item_pitchfork" },
			{ "static/planningtable.tik", "static_furniture_planningtable" },
			{ "static/plunger.tik", "static_item_plunger" },
			{ "static/pots.tik", "static_furniture_pots" },
			{ "static/pounder.tik", "static_item_pounder" },
			{ "static/produce_cart.tik", "static_farm_producecart" },
			{ "static/projector.tik", "static_projector" },
			{ "static/protractor.tik", "static_item_protractor" },
			{ "static/ramp_quadleg.tik", "static_obstacle_ramp-quadleg" },
			{ "static/ramp_tripleleg.tik", "static_obstacle_ramp-tripleleg" },
			{ "static/rations_sh.tik", "static_item_rations-sh" },
			{ "static/rock_large.tik", "static_natural_rock_plain-large" },
			{ "static/rock_medium.tik", "static_natural_rock_plain-medium" },
			{ "static/rock_winter_large.tik", "static_natural_rock_winter-large" },
			{ "static/rock_winter_medium.tik", "static_natural_rock_winter-medium" },
			{ "static/rolltop_desk.tik", "static_furniture_rolltopdesk" },
			{ "static/round_table.tik", "static_furniture_round-table" },
			{ "static/rubble_bigpile.tik", "static_obstacle_rubble-bigpile" },
			{ "static/rubble_smallpile.tik", "static_obstacle_rubble-smallpile" },
			{ "static/sandbag_large_semicircle.tik", "static_obstacle_sandbag-large-semicircle" },
			{ "static/sandbag_large_semicircle_winter.tik", "static_obstacle_sandbag-large-semicircle-winter" },
			{ "static/sandbag_link_bottomcap.tik", "static_obstacle_sandbag-link-bottomcap" },
			{ "static/sandbag_link_bottomcap_winter.tik", "static_obstacle_sandbag-link-bottomcap-winter" },
			{ "static/sandbag_link_leftbend.tik", "static_obstacle_sandbag-link-leftbend" },
			{ "static/sandbag_link_main.tik", "static_obstacle_sandbag-link-main" },
			{ "static/sandbag_link_main_winter.tik", "static_obstacle_sandbag-link-main-winter" },
			{ "static/sandbag_link_rightbend.tik", "static_obstacle_sandbag-link-rightbend" },
			{ "static/sandbag_link_rightbend_winter.tik", "static_obstacle_sandbag-link-rightbend-winter" },
			{ "static/sandbag_link_topcap.tik", "static_obstacle_sandbag-link-topcap" },
			{ "static/sandbag_link_topcap_winter.tik", "static_obstacle_sandbag-link-topcap-winter" },
			{ "static/sandbag_longsegment.tik", "static_obstacle_sandbag-longsegment" },
			{ "static/sandbag_rightangle.tik", "static_obstacle_sandbag-rightangle" },
			{ "static/sandbag_small_semicircle.tik", "static_obstacle_sandbag-small-semicircle" },
			{ "static/sandbag_small_semicircle_winter.tik", "static_obstacle_sandbag-small-semicircle-winter" },
			{ "static/scr536.tik", "static_items_scr536-walkietalkie" },
			{ "static/screwdriver.tik", "static_item_screwdriver" },
			{ "static/servingplate.tik", "static_item_servingplate" },
			{ "static/shovel.tik", "static_item_shovel" },
			{ "static/signpost_caen.tik", "static_signpost_caen" },
			{ "static/signpost_stlo.tik", "static_signpost_stlo" },
			{ "static/signpost_veret.tik", "static_signpost_veret" },
			{ "static/signpost_vierville.tik", "static_signpost_vierville" },
			{ "static/simpledesk.tik", "static_furniture_simpledesk" },
			{ "static/simpledesk_ns.tik", "static_furniture_simpledesk" },
			{ "static/simplemetaldesk.tik", "static_furniture_simplemetaldesk" },
			{ "static/simplerectable.tik", "static_simplerectable" },
			{ "static/singlemetalbed.tik", "static_furniture_singlemetalbed" },
			{ "static/single_bed.tik", "static_furniture_singlebed" },
			{ "static/single_bed_damaged.tik", "static_furniture_singlebed_damaged" },
			{ "static/sliding_ruler.tik", "static_item_sliding-ruler" },
			{ "static/smallfilecabinet.tik", "static_furniture_smallfilecabinet" },
			{ "static/smallhutch.tik", "static_furniture_smallhutch" },
			{ "static/small_pitcher.tik", "static_decor_mug-metal" },
			{ "static/south_africa_ceramic_pot_1.tik", "static_item_south-africa-ceramic-pot-1" },
			{ "static/south_africa_ceramic_pot_2.tik", "static_item_south-africa-ceramic-pot-2" },
			{ "static/south_africa_ceramic_pot_3.tik", "static_item_south-africa-ceramic-pot-3" },
			{ "static/south_africa_ceramic_pot_4.tik", "static_item_south-africa-ceramic-pot-4" },
			{ "static/south_africa_ceramic_pot_5.tik", "static_item_south-africa-ceramic-pot-5" },
			{ "static/south_africa_ceramic_pot_6.tik", "static_item_south-africa-ceramic-pot-6" },
			{ "static/south_africa_ceramic_pot_7.tik", "static_item_south-africa-ceramic-pot-7" },
			{ "static/speakerbox1.tik", "static_uboat_speakerbox1" },
			{ "static/square_table.tik", "static_furniture_square-table" },
			{ "static/square_table_red.tik", "static_furniture_square-table-red" },
			{ "static/stackedshelves.tik", "static_furniture_stackedshelves" },
			{ "static/static_6lamp.tik", "static_uboat_6lamp-panel" },
			{ "static/static_airtank.tik", "static_uboat_airtank" },
			{ "static/static_alarmbell.tik", "static_miscobj_alarmbell" },
			{ "static/static_atlas-echolot.tik", "static_uboat_atlas-echolot" },
			{ "static/static_bangalore.tik", "static_item_bangalore" },
			{ "static/static_barometer.tik", "static_uboat_bridge-barometer" },
			{ "static/static_bigswitch1.tik", "static_uboat_bigswitch1" },
			{ "static/static_bigswitch2.tik", "static_uboat_bigswitch2" },
			{ "static/static_cablespool_empty.tik", "static_equipment_cablespool-empty" },
			{ "static/static_cablespool_full.tik", "static_equipment_cablespool-full" },
			{ "static/static_cagelight.tik", "static_lamp_bunker-cagelight" },
			{ "static/static_calendar.tik", "static_uboat_calendar" },
			{ "static/static_clock1.tik", "static_furniture_clock1" },
			{ "static/static_clock1a.tik", "static_furniture_clock1-roman" },
			{ "static/static_clock2.tik", "static_furniture_clock2" },
			{ "static/static_clock3.tik", "static_furniture_clock3" },
			{ "static/static_dieselrpm.tik", "static_uboat_dieselrpm-gauge" },
			{ "static/static_domelight.tik", "static_lamp_bunker-domelight" },
			{ "static/static_e-light.tik", "static_uboat_emergency-lamp" },
			{ "static/static_electricbox1.tik", "static_uboat_electricbox1" },
			{ "static/static_enginecover.tik", "static_uboat_engine-cover" },
			{ "static/static_enigmaclosed.tik", "static_uboat_enigmaclosed" },
			{ "static/static_enigmaopen.tik", "static_uboat_enigmaopen" },
			{ "static/static_eot.tik", "static_uboat_engine-order-telegraph1" },
			{ "static/static_eot2.tik", "static_uboat_engine-order-telegraph2" },
			{ "static/static_germanhelmet_wehrmact.tik", "static_equipment_german-helmet-wehrmact" },
			{ "static/static_gyrocompass.tik", "static_uboat_gyrocompass" },
			{ "static/static_helmcontrol1.tik", "static_uboat_helmcontrol1" },
			{ "static/static_helmcontrol2.tik", "static_uboat_helmcontrol2" },
			{ "static/static_kriegsflag.tik", "static_flags_kriegsflag" },
			{ "static/static_m1_garand.tik", "static_weapon_allied_m1-garand" },
			{ "static/static_nazibanner.tik", "static_flags_nazibanner1" },
			{ "static/static_nazibanner1a.tik", "static_flags_nazibanner1a" },
			{ "static/static_nazibanner1b.tik", "static_flags_nazibanner1b" },
			{ "static/static_nazibanner2.tik", "static_flags_nazibanner2" },
			{ "static/static_naziflag1.tik", "static_flags_naziflag1" },
			{ "static/static_naziflag2.tik", "static_flags_naziflag2" },
			{ "static/static_panelknob-big.tik", "static_uboat_panelknob-big" },
			{ "static/static_panelknob-small.tik", "static_uboat_panelknob-small" },
			{ "static/static_periscope.tik", "static_uboat_periscope" },
			{ "static/static_powerbox1.tik", "static_uboat_tan-power-box" },
			{ "static/static_powerbox2.tik", "static_uboat_tan-power-box2" },
			{ "static/static_powerpanel.tik", "static_uboat_powerpanel" },
			{ "static/static_radio2.tik", "static_uboat_radio2" },
			{ "static/static_radiostation1.tik", "static_uboat_radiostation1" },
			{ "static/static_radiostation2.tik", "static_uboat_radiostation2" },
			{ "static/static_radiostation3.tik", "static_uboat_radiostation3" },
			{ "static/static_radiostation4.tik", "static_uboat_radiostation4" },
			{ "static/static_ruddergauge1.tik", "static_uboat_ruddergauge1" },
			{ "static/static_ruddergauge2.tik", "static_uboat_ruddergauge2" },
			{ "static/static_subalarmbell.tik", "static_uboat_subalarmbell" },
			{ "static/static_subclock.tik", "static_uboat_subclock" },
			{ "static/static_subhatch.tik", "static_uboat_subhatch" },
			{ "static/static_subradio1.tik", "static_uboat_subradio1" },
			{ "static/static_substool.tik", "static_uboat_substool" },
			{ "static/static_subvalve-big-black1.tik", "static_uboat_subvalve-big-black1" },
			{ "static/static_subvalve-big-black2.tik", "static_uboat_subvalve-big-black2" },
			{ "static/static_subvalve-big-grey1.tik", "static_uboat_subvalve-big-grey1" },
			{ "static/static_subvalve-big-grey2.tik", "static_uboat_subvalve-big-grey2" },
			{ "static/static_subvalve-big-red.tik", "static_uboat_subvalve-big-red" },
			{ "static/static_subvalve-med-black1.tik", "static_uboat_subvalve-med-black1" },
			{ "static/static_subvalve-med-black2.tik", "static_uboat_subvalve-med-black2" },
			{ "static/static_subvalve-med-grey1.tik", "static_uboat_subvalve-med-grey1" },
			{ "static/static_subvalve-med-grey2.tik", "static_uboat_subvalve-med-grey2" },
			{ "static/static_subvalve-med-red.tik", "static_uboat_subvalve-med-red" },
			{ "static/static_subvalve-small-black1.tik", "static_uboat_subvalve-small-black1" },
			{ "static/static_subvalve-small-black2.tik", "static_uboat_subvalve-small-black2" },
			{ "static/static_subvalve-small-grey1.tik", "static_uboat_subvalve-small-grey1" },
			{ "static/static_subvalve-small-grey2.tik", "static_uboat_subvalve-small-grey2" },
			{ "static/static_subvalve-small-red.tik", "static_uboat_subvalve-small-red" },
			{ "static/static_sub_bunk.tik", "static_furniture_sub-bunk" },
			{ "static/static_tempgauge.tik", "static_uboat_engine-temperature-gauge" },
			{ "static/static_thompsonsmg.tik", "static_weapon_allied_thompson" },
			{ "static/static_tiefenmesser.tik", "static_uboat_tiefenmesser-gauge" },
			{ "static/static_torpedo.tik", "static_uboat_torpedo" },
			{ "static/static_us-helmet_29th-net.tik", "static_equipment_us-helmet29th-net" },
			{ "static/static_us-helmet_29th.tik", "static_equipment_us-helmet29th" },
			{ "static/static_us-helmet_captain.tik", "static_equipment_us-helmetcaptain" },
			{ "static/static_us-helmet_engineer01.tik", "static_equipment_us-engineer01" },
			{ "static/static_us-helmet_engineer02.tik", "static_equipment_us-engineer02" },
			{ "static/static_us-helmet_ltnt.tik", "static_equipment_us-helmet-ltnt" },
			{ "static/static_us-helmet_medic.tik", "static_equipment_us-helmet-medic" },
			{ "static/static_us-helmet_private_net_cig.tik", "static_equipment_us-private-net-cig" },
			{ "static/static_us-helmet_sergeant.tik", "static_equipment_us-helmet-sergeant" },
			{ "static/static_ushelmet_private.tik", "static_equipment_us-helmet-private" },
			{ "static/static_ushelmet_private_net.tik", "static_equipment_us-helmet-private-net" },
			{ "static/static_valvetree.tik", "static_uboat_valvetree" },
			{ "static/steilhangrenate.tik", "static_weapon_steilhangrenate" },
			{ "static/switch1.tik", "static_item_switch1" },
			{ "static/switch2.tik", "static_item_switch2" },
			{ "static/tablewithumbrella.tik", "static_tablewithumbrella" },
			{ "static/tent.tik", "static_furniture_tent" },
			{ "static/toilet.tik", "static_furniture_toilet-bowl-and-tank" },
			{ "static/toilet_short.tik", "static_toilet_short" },
			{ "static/toolbox_closed.tik", "static_item_toolbox-closed" },
			{ "static/toolbox_opened.tik", "static_item_toolbox-opened" },
			{ "static/towel.tik", "static_item_toolbox-opened" },
			{ "static/townbell.tik", "static_town_townbell" },
			{ "static/tree_birch.tik", "static_natural_tree_birch" },
			{ "static/tree_commontree.tik", "static_natural_tree_commontree" },
			{ "static/tree_juniper.tik", "static_natural_tree_juniper" },
			{ "static/tree_oak.tik", "static_natural_tree_oak" },
			{ "static/tree_regularpalm.tik", "static_natural_tree_regularpalm" },
			{ "static/tree_shortpalm.tik", "static_natural_tree_shortpalm" },
			{ "static/tree_smallyucca.tik", "static_natural_tree_smallyucca" },
			{ "static/tree_squatpalm.tik", "static_natural_tree_squatpalm" },
			{ "static/tree_stump.tik", "static_natural_tree_stump" },
			{ "static/tree_tallyucca.tik", "static_natural_tree_tallyucca" },
			{ "static/tree_winter_fillpine.tik", "static_natural_tree_winter-fillpine" },
			{ "static/tree_winter_midpine.tik", "static_natural_tree_winter-midpine" },
			{ "static/tree_winter_smallpine.tik", "static_natural_tree_winter-smallpine" },
			{ "static/tree_winter_tallpine.tik", "static_natural_tree_winter-tallpine" },
			{ "static/tree_winter_thicktrunk.tik", "static_natural_tree_winter-thicktrunk" },
			{ "static/tree_winter_thintrunk.tik", "static_natural_tree_winter-thintrunk" },
			{ "static/trunk.tik", "static_item_trunk" },
			{ "static/typewriter.tik", "static_furniture_typewriter" },
			{ "static/typewriter_ns.tik", "static_furniture_typewriter" },
			{ "static/uboat.tik", "static_vehicles_uboat" },
			{ "static/upperhull.tik", "static_vehicles_uboat-upperhull-U529" },
			{ "static/upperhull2.tik", "static_vehicles_uboat-upperhull-Uboat2" },
			{ "static/upperhull3.tik", "static_vehicles_uboat-upperhull-Uboat3" },
			{ "static/vanity.tik", "static_furniture_vanity" },
			{ "static/vehicle_bmwbike.tik", "static_vehicle_german_bmwbike" },
			{ "static/vehicle_c47.tik", "static_vehicle_allied_airplane-c47" },
			{ "static/vehicle_car_burnt.tik", "static_vehicle_europe_car-burnt" },
			{ "static/vehicle_car_rusted.tik", "static_vehicle_europe_car-rusted" },
			{ "static/vehicle_dtruck_burnt.tik", "static_vehicle_europe_deliverytruck-burnt" },
			{ "static/vehicle_dtruck_rusted.tik", "static_vehicle_europe_deliverytruck-rusted" },
			{ "static/vehicle_european_car_grey.tik", "static_vehicle_europe_car-grey" },
			{ "static/vehicle_european_car_silver.tik", "static_vehicle_europe_car-silver" },
			{ "static/vehicle_european_car_tan.tik", "static_vehicle_europe_car-tan" },
			{ "static/vehicle_european_delivery_truck_green.tik", "static_vehicle_europe_deliverytruck-green" },
			{ "static/vehicle_european_delivery_truck_grey.tik", "static_vehicle_europe_deliverytruck-grey" },
			{ "static/vehicle_european_delivery_truck_red3.tik", "static_vehicle_europe_deliverytruck-red" },
			{ "static/vehicle_fockwulf.tik", "static_vehicle_german_airplane-fockwulf" },
			{ "static/vehicle_gmctruck.tik", "static_vehicle_allied_gmctruck" },
			{ "static/vehicle_higgins.tik", "static_vehicle_allied_higgins-boat" },
			{ "static/vehicle_jeep.tik", "static_vehicle_allied_jeep" },
			{ "static/vehicle_ktigertank.tik", "static_vehicle_german_tank-kingtiger" },
			{ "static/vehicle_m3.tik", "static_vehicle_allied_m3halftrack" },
			{ "static/vehicle_mercedes.tik", "static_vehicle_german_mercedes" },
			{ "static/vehicle_opeltruck.tik", "static_vehicle_german_opeltruck" },
			{ "static/vehicle_opeltruck_green.tik", "static_vehicle_german_opeltruck-green" },
			{ "static/vehicle_panzer_iv_europe.tik", "static_vehicle_german_panzer-iv-europe" },
			{ "static/vehicle_sdkfz.tik", "static_vehicle_german_sdkfz" },
			{ "static/vehicle_shermantank.tik", "static_vehicle_allied_tank-sherman" },
			{ "static/vehicle_shermantank_dead.tik", "static_vehicle_allied_tank-sherman-destroyed" },
			{ "static/vehicle_stuka.tik", "static_vehicle_german_airplane-stuka" },
			{ "static/vehicle_tigertank.tik", "static_vehicle_german_tank-tiger" },
			{ "static/vehicle_tigertankdsrt.tik", "static_vehicle_german_tank-tiger-desert" },
			{ "static/vent_valve.tik", "static_item_vent_valve" },
			{ "static/wagon.tik", "static_farm_wagon" },
			{ "static/wardrobe.tik", "static_furniture_wardrobe" },
			{ "static/wardrobe_damaged.tik", "static_furniture_wardrobe-damaged" },
			{ "static/welding_torch.tik", "static_equipment_welding-torch" },
			{ "static/whiteplate.tik", "static_item_whiteplate" },
			{ "static/wicker_basket_1.tik", "static_item_wicker-basket-1" },
			{ "static/wicker_basket_2.tik", "static_item_wicker-basket-2" },
			{ "static/wicker_basket_3.tik", "static_item_wicker-basket-3" },
			{ "static/widepainting.tik", "static_furniture_widepainting" },
			{ "static/winecasks.tik", "static_furniture_winecasks" },
			{ "static/wirerailholder.tik", "static_obstacle_wirerailholder" },
			{ "static/woodbucket.tik", "static_item_woodbucket" },
			{ "static/woodenspoon.tik", "static_item_woodenspoon" },
			{ "static/worklamp.tik", "static_item_worklamp" },
			{ "static/worktable.tik", "static_furniture_worktable" },
			{ "static/worndowntable.tik", "static_furniture_worndowntable" },
			{ "static/wrenches.tik", "static_item_wrenches" },
			{ "static/w_parkbench.tik", "static_furniture_w_parkbench" },
			{ "statweapons/mg42_bipod.tik", "static_weapon_mg42bipod" },
			{ "vehicles/panzer_iv.tik", "static_vehicle_german_panzer_iv" },
			{ "vehicles/static_higginsdoor.tik", "static_vehicle_allied_higgins-boat-door" },
			{ "vehicles/vehicle_car_burnt-notstatic.tik", "static_vehicle_europe_car-burnt-nonstatic" },
			{ "weapons/static_kar98.tik", "static_weapon_german_kar98" },
			{ "ammo/stuka_bomb.tik", "addon_static_props_bomb" },
			{ "animate/1_1_omnitree1_nodraw.tik", "addon_static_natural_fallingtrees_tree07_nodraw" },
			{ "animate/1_1_omnitree2_nodraw.tik", "addon_static_natural_fallingtrees_tree08_nodraw" },
			{ "animate/falling_tree01_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree01_nodraw" },
			{ "animate/falling_tree02_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree02_nodraw" },
			{ "animate/falling_tree03_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree03_nodraw" },
			{ "animate/falling_tree04_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree04_nodraw" },
			{ "animate/falling_tree05_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree05_nodraw" },
			{ "animate/falling_tree06_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree06_nodraw" },
			{ "animate/falling_tree07_oriented_nodraw.tik", "addon_static_natural_fallingtrees_tree07a_nodraw" },
			{ "animate/tree_commontree_nodraw.tik", "addon_static_natural_tree_static_nodraw" },
			{ "lights/lightbulb_caged_green.tik", "addon_static_lamp_lightbulb-caged-green" },
			{ "natural/grassclod.tik", "addon_static_natural_grassclod" },
			{ "natural/tabush_winter_fullshrub.tik", "addon_static_natural_bushes_winter-fullshrub" },
			{ "natural/tree2_winter.tik", "addon_static_natural_tree_tree-skinnywinter" },
			{ "natural/tree_winter_midpine_nosprite.tik", "addon_static_natural_tree_winter-midpinenosprite" },
			{ "natural/tree_winter_smallpine_nosprite.tik", "addon_static_natural_tree_winter-smallpinenosprite" },
			{ "natural/tree_winter_tallpine_nosprite.tik", "addon_static_natural_tree_winter-tallpinenosprite" },
			{ "static/1_1-parachute-barn.tik", "addon_static_props_parachute-barn" },
			{ "static/1_3_bridge.tik", "addon_animate_1-3_bridge_static" },
			{ "static/bmw_destroyed.tik", "addon_static_bmw-destroyed" },
			{ "static/c47_interior.tik", "addon_static_props_C47-Interior" },
			{ "static/control_panel.tik", "addon_static_fuel-control-panel" },
			{ "static/dead_paratrooper.tik", "addon_static_dead-paratrooper" },
			{ "static/fuel_coupler.tik", "addon_static_fuel-coupler" },
			{ "static/fuel_coupler_d.tik", "addon_static_fuel-coupler-destroyed" },
			{ "static/fuel_tank.tik", "addon_static_props_fuel-tank" },
			{ "static/fuel_tanks.tik", "addon_static_fuel-tanks" },
			{ "static/fuel_tanks_d.tik", "addon_static_fuel-tanks-destroyed" },
			{ "static/fuel_tank_d.tik", "addon_static_props_fuel-tank-destroyed" },
			{ "static/generator.tik", "addon_static_generator" },
			{ "static/granatwerfer.tik", "addon_static_props_granatwerfer-mortar" },
			{ "static/groundportal_1_1.tik", "addon_static_groundPortal-t1l1" },
			{ "static/haystack1.tik", "addon_static_props_haystack1" },
			{ "static/haystack_dry01.tik", "addon_static_props_haystack-dry-01" },
			{ "static/haystack_dry02.tik", "addon_static_props_haystack-dry-02" },
			{ "static/hedgehog_snowy.tik", "addon_static_obstacle_hedgehog-snowy" },
			{ "static/logs0.tik", "addon_static_natural_logs_logs0" },
			{ "static/logstack0.tik", "addon_static_natural_logs_logStack0" },
			{ "static/officer_tent.tik", "addon_static_props_officer-tent" },
			{ "static/parachute.tik", "addon_static_props_parachute" },
			{ "static/parachute_actors.tik", "addon_static_props_parachute" },
			{ "static/rock1.tik", "addon_static_natural_rocks_rock1" },
			{ "static/rock2.tik", "addon_static_natural_rocks_rock2" },
			{ "static/rusty_truck.tik", "addon_static_vehicle_rusty-truck" },
			{ "static/sandbag_small_semicircle_winter2.tik", "addon_static_props_sandbags-winter" },
			{ "static/snobranch1.tik", "addon_static_natural_branches_snobranch1" },
			{ "static/snobranch2.tik", "addon_static_natural_branches_snowbranch2" },
			{ "static/snobranch3.tik", "addon_static_natural_branches_snowbranch3" },
			{ "static/snowbush0.tik", "addon_static_natural_bushes_snowbush0" },
			{ "static/snowbush1.tik", "addon_static_natural_bushes_snowbush1" },
			{ "static/snowcanopy.tik", "addon_static_props_SnowCanopy" },
			{ "static/snowtent.tik", "addon_static_props_SnowTent" },
			{ "static/snowtree0.tik", "addon_static_natural_tree_SnowTree0" },
			{ "static/snowtree1.tik", "addon_static_natural_tree_SnowTree1" },
			{ "static/snowtree2.tik", "addon_static_natural_tree_SnowTree2" },
			{ "static/snowtree3.tik", "addon_static_natural_tree_SnowTree3" },
			{ "static/snowtree4.tik", "addon_static_natural_tree_SnowTree4" },
			{ "static/snowtree5.tik", "addon_static_natural_tree_SnowTree5" },
			{ "static/snowtree6.tik", "addon_static_natural_tree_SnowTree6" },
			{ "static/snowtree7.tik", "addon_static_natural_tree_SnowTree7" },
			{ "static/snowtree8.tik", "addon_static_natural_tree_SnowTree8" },
			{ "static/snowycrate.tik", "addon_static_props_snowcrate2" },
			{ "static/snow_crate.tik", "addon_static_props_snowcrate" },
			{ "static/static_nazieagle.tik", "addon_static_props_flags_nazieagle" },
			{ "static/static_nazieagle2.tik", "addon_static_props_flags_nazieagle2" },
			{ "static/supplydropcrate.tik", "addon_static_props_supplyDropCrate" },
			{ "static/upperhull_ng.tik", "addon_static_vehicle_uboat-upperhull-NoGuns" },
			{ "static/natural/grassclod.tik", "static_natural_grass" },
			{ "static/natural/tabush_winter_fullshrub.tik", "static_natural_TAbush_winter-fullshrub" },
			{ "static/natural/tree2_winter.tik", "static_natural_TAtree_skinnywinter" },
			{ "static/natural/tree_winter_midpine_nosprite.tik", "static_natural_tree_winter-midpinenosprite" },
			{ "static/natural/tree_winter_smallpine_nosprite.tik", "static_natural_tree_winter-smallpinenosprite" },
			{ "static/natural/tree_winter_tallpine_nosprite.tik", "static_natural_tree_winter-tallpinenosprite" },
			{ "fx/corona_orange.tik", "nonstatic_e2l2_corona-orange" },
			{ "items/af_p_lantern.tik", "addon_static_Lantern" },
			{ "lights/candleholder.tik", "addon_static_lamp_Candleholder" },
			{ "lights/candle_fell.tik", "addon_static_lamp_Candle_Fell" },
			{ "lights/candle_standing_m.tik", "addon_static_lamp_Candle_Standing-Medium" },
			{ "lights/candle_standing_s.tik", "addon_static_lamp_Candle_Standing-Short" },
			{ "lights/candle_standing_t.tik", "addon_static_lamp_Candle_Standing-Tall" },
			{ "lights/standinglight.tik", "addon_static_lamp_StandingLight" },
			{ "lights/walllight.tik", "addon_static_lamp_WallLight" },
			{ "lights/wallsconce-frosted_off.tik", "static_lamp_wallsconce-frosted_off" },
			{ "miscobj/crane_grey.tik", "static_equipment_crane" },
			{ "miscobj/lightbulb_covered_nowire.tik", "nonstatic_e2l2_lightbulb" },
			{ "miscobj/pulse_detonator.tik", "static_equipment_detonator-pulsating" },
			{ "miscobj/radio_military_pulse.tik", "nonstatic_equipment_radio-military-pulse" },
			{ "miscobj/sc_p_campfire.tik", "addon_static_props_Sc_P_CampFire" },
			{ "static/af_phanger_dest.tik", "addon_static_AF_Phanger_dest" },
			{ "static/af_p_crane.tik", "addon_static_Crane" },
			{ "static/af_p_foldeduni.tik", "addon_static_Folded-Italian-Uniform" },
			{ "static/af_p_lifepreserver.tik", "addon_static_LifePreserver" },
			{ "static/af_p_shrub.tik", "static_natural_bush_E1shrub" },
			{ "static/af_p_stretcher.tik", "addon_static_Stretcher" },
			{ "static/af_p_tent_damaged.tik", "addon_static_Tent_Damaged" },
			{ "static/af_v_boat_1.tik", "addon_static_Boat-1" },
			{ "static/af_v_boat_2.tik", "addon_static_Boat-2" },
			{ "static/af_v_boat_3.tik", "addon_static_Boat-3" },
			{ "static/corona_bologna.tik", "static_corona_bologna" },
			{ "static/door_frame.tik", "addon_static_motif_malta-door-frame" },
			{ "static/e3l3lamp.tik", "addon_static_lamp_e3l3lamp" },
			{ "static/it_p_k5schematics.tik", "addon_static_K5Schematics" },
			{ "static/it_p_radarnowood.tik", "addon_static_RadarStation-NoWood" },
			{ "static/it_p_radarstation.tik", "addon_static_RadarStation-Dark" },
			{ "static/it_p_statue.tik", "addon_static_Statue" },
			{ "static/lightbulb_covered_nowire2.tik", "static_lamp_lightbulb-covered-nowire2" },
			{ "static/lightbulb_covered_nowire3.tik", "static_lamp_lightbulb-covered-onwire3" },
			{ "static/maltalantern.tik", "addon_static_motif_malta-lantern" },
			{ "static/maltalanternchain.tik", "addon_static_motif_malta-lantern-chain" },
			{ "static/maltasign.tik", "addon_static_motif_malta-sign-3crowns" },
			{ "static/maltasign2.tik", "addon_static_motif_malta-sign-wildgoose" },
			{ "static/maltastatue.tik", "addon_static_motif_malta-statue" },
			{ "static/montecassino_groundportal.tik", "addon_static_MonteCassino_GroundPortal" },
			{ "static/mp_p_minecart.tik", "addon_static_Minecart" },
			{ "static/parachute2_3.tik", "addon_static_props_parachute2_3" },
			{ "static/rock_large_brown.tik", "addon_static_natural_rock_Brown-large" },
			{ "static/rock_medium_brown.tik", "addon_static_natural_rock_Brown-medium" },
			{ "static/sc_p_hangingchute.tik", "addon_static_props_sc_p_hangingchute" },
			{ "static/sc_p_map.tik", "addon_static_map" },
			{ "static/sc_p_pickaxe.tik", "addon_static_PickAxe" },
			{ "static/sc_p_shovel.tik", "addon_static_Shovel" },
			{ "static/static_cagelight_shadow.tik", "static_lamp_bunker-cagelight-shadow" },
			{ "static/tent_e2l1_new.tik", "addon_static_tente1l1new" },
			{ "static/tent_medic2.tik", "addon_static_tentmedic2" },
			{ "static/tow_p_saintb.tik", "addon_static_SaintB" },
			{ "static/tow_p_sainte.tik", "addon_static_SaintE" },
			{ "lights/swinging_lantern.tik", "lamp_swinging-lantern" },
		};

	}
}
