/*
    Copyright (C) 2008 Rouslan Dimitrov

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
 The thread pool parallizes a 'for loop'. RunOn(f, a, s, e) is semantically equivalent to:
 [run in #cpu threads in parallel] for(int i=s; i<e; i++) f(i, a);

 The thread pool runs asynchronously and needs to be explicitly
 serialized by WaitThreads() which returns only after the work is complete.

 Usage:
 1. pool.RunOn(...); // May return before work is complete
 2. Optional work independent of the data used for the computation in 1.
 3. pool.WaitThreads();
*/

#pragma once

#ifndef _WIN32
#error This file can only be compiled on a Win32 platform
#endif

#define WIN32_LEAN_AND_MEAN

// On my VS 2005 on Windows XP, _WIN32_WINNT is not defined
#ifndef _WIN32_WINNT
#define _WIN32_WINNT 0x500	// Windows 2000
#endif

#include <windows.h>

const int MAX_THREADS = 8;	// should be less than the number of bits in DWORD

typedef void (*WORKFUNC)(int, void*);

int __fastcall AtomicXAdd(int *, int);
int __fastcall GetCurrentProcessorNumberWinAll(void);

class CThreadPool {
public:
	void		Init(bool useCallingThread);
	void		RunOn(WORKFUNC func, void* arg, int startIndex, int endIndex);
	void		ShutDown();
	void		WaitThreads();

private:
	void		WaitThreadsToExit();

	friend
	DWORD WINAPI ThreadScheduler(LPVOID lpThreadPool);

	friend void	DoWork(CThreadPool *pool);

	HANDLE		hThreads[MAX_THREADS];
	HANDLE		hThreadsDone[MAX_THREADS];
	HANDLE		hMoreWork;
	WORKFUNC	function;
	void*		argument;
	int			curIndex;
	int			endIndex;
	int			nThreads;
	bool		useCallingThread;
	bool		bRunning;
};

inline void CThreadPool::WaitThreads() {
	WaitForMultipleObjects(nThreads, hThreadsDone, true, INFINITE);
}

inline void CThreadPool::WaitThreadsToExit() {
	WaitForMultipleObjects(nThreads, hThreads, true, INFINITE);
}